const {
  default: makeWASocket,
  DisconnectReason,
  fetchLatestBaileysVersion,
  useMultiFileAuthState,
} = require("@whiskeysockets/baileys");
const { Boom } = require("@hapi/boom");
const pino = require("pino");
const path = require("path");
const fs = require("fs");
const http = require("http");
const express = require("express");
const fileUpload = require("express-fileupload");
const cors = require("cors");
const qrcode = require("qrcode");
const socketIo = require("socket.io");
const axios = require("axios");

// Load environment variables
require('dotenv').config({ path: path.join(__dirname, '.env') });

// Configuration
const CONFIG = {
  PORT: process.env.PORT || 7001,
  NODE_ENV: process.env.NODE_ENV || 'production',
  BOT_NAME: process.env.BOT_NAME || 'wabt-bly-lc-mac',
  BOT_VERSION: process.env.BOT_VERSION || '2.0.0-Baileys',
  API_KEY: process.env.API_KEY || 'Q168sRgWlptFxv6RBRTbYf75oAzFEfspkWqCVIYjh2J',
  GOOGLE_SHEETS_DEPLOY_ID: process.env.GOOGLE_SHEETS_DEPLOY_ID || 'AKfycbyf4QHMhzx3OL2BqdJ1E9uXJ5mcxVv4oNKr06oOrHIGO7z4kjEFuxhSG2z3zoh9sAKs',
  CPANEL_MODE: process.env.CPANEL_MODE === 'true',
  LOG_LEVEL: process.env.LOG_LEVEL || 'info'
};

// Express setup
const app = express();
const server = http.createServer(app);
const io = socketIo(server, {
  cors: {
    origin: "*",
    methods: ["GET", "POST"]
  }
});
const port = CONFIG.PORT;

// Middleware
app.use(fileUpload({
    createParentPath: true,
    limits: { fileSize: 50 * 1024 * 1024 }, // 50MB max file size
}));
app.use(cors());
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));
app.use("/assets", express.static(path.join(__dirname, "client/assets")));

// Routes
app.get("/scan", (req, res) => {
    res.sendFile("./client/server.html", {
        root: __dirname,
    });
});

app.get("/", async (req, res) => {
    res.sendFile("./client/index.html", {
        root: __dirname,
    });
});

// Global variables
let sock;
let qrDinamic;
let soket;
let isConnected = false;

// Cache untuk menyimpan mapping participant ke nomor HP
const participantCache = new Map();

// Cache untuk menyimpan nama participant dari pesan yang masuk
const participantNamesCache = new Map();

// Nomor admin yang diizinkan untuk command tertentu
const allowedNumbers = ['6285750506009', '6281944386646', '6281369807709'];

// Daftar titik-titik lokasi yang diizinkan
const allowedLocations = [
    { lat: 0.19880519469882046, lon: 111.46750166076582, name: "Lokasi 1" },
    { lat: 0.2068080, lon: 111.3978390, name: "Lokasi 2" },
    { lat: 0.1764250, lon: 111.4852220, name: "Lokasi 3" },
    { lat: 0.2385110, lon: 111.4011650, name: "Lokasi 4" },
    { lat: 0.1422870, lon: 111.5584110, name: "Lokasi 5" },
    { lat: 0.2020360, lon: 111.5017530, name: "Lokasi 6" },
    { lat: 0.1764222, lon: 111.3599731, name: "Lokasi 7" },
    { lat: 0.1470750, lon: 111.5048780, name: "Lokasi 8" },
    { lat: 0.2034880, lon: 111.4313150, name: "Lokasi 9" },
    { lat: 0.233997, lon: 111.404545, name: "Lokasi 10" },
    { lat: 0.2335450, lon: 111.4044350, name: "Lokasi 11" },
    { lat: 0.13409, lon: 111.59596, name: "Lokasi 12" },
    { lat: 0.176026, lon: 111.358402, name: "Lokasi 13" },
    { lat: -0.0555778, lon: 109.3209126, name: "Lokasi 14" }
];

async function connectToWhatsApp() {
  try {
    const { state, saveCreds } = await useMultiFileAuthState("baileys_auth_info");

    const { version, isLatest } = await fetchLatestBaileysVersion();
    console.log(`Using WA v${version.join(".")}, isLatest: ${isLatest}`);

    sock = makeWASocket({
      version,
      logger: pino({ level: "silent" }),
      printQRInTerminal: true,
      auth: state,
      browser: ["wabt-bly-lc-mac", "Chrome", "2.0.0"],
      syncFullHistory: false,
      generateHighQualityLinkPreview: true,
      markOnlineOnConnect: false,
      keepAliveIntervalMs: 30000,
      connectTimeoutMs: 60000,
      defaultQueryTimeoutMs: 60000,
      emitOwnEvents: true,
      fireInitQueries: true,
      msgRetryCounterCache: undefined,
    });

    sock.multi = true;

    sock.ev.on("connection.update", async (update) => {
    const { connection, lastDisconnect, qr } = update;
    qrDinamic = qr;

    // Helper: hapus semua file di folder auth jika session error
    function deleteAuthFiles() {
      const authDir = path.join(__dirname, 'baileys_auth_info');
      if (fs.existsSync(authDir)) {
        fs.readdirSync(authDir).forEach(file => {
          const filePath = path.join(authDir, file);
          try {
            fs.unlinkSync(filePath);
            console.log(`Deleted auth file: ${filePath}`);
          } catch (err) {
            console.error(`Failed to delete auth file: ${filePath}`, err);
          }
        });
      }
    }

    if (connection === "close") {
      let reason = new Boom(lastDisconnect?.error)?.output.statusCode;
      if (reason === DisconnectReason.badSession) {
        console.log("Bad Session File, Please Delete Session and Scan Again");
        // deleteAuthFiles();
        sock.logout();
      } else if (reason === DisconnectReason.connectionClosed) {
        console.log("Connection closed, reconnecting....");
        connectToWhatsApp();
      } else if (reason === DisconnectReason.connectionLost) {
        console.log("Connection Lost from Server, reconnecting...");
        connectToWhatsApp();
      } else if (reason === DisconnectReason.connectionReplaced) {
        console.log("Connection Replaced, Another New Session Opened, Please Close Current Session First");
        // deleteAuthFiles();
        sock.logout();
      } else if (reason === DisconnectReason.loggedOut) {
        console.log("Device Logged Out, Please Delete Session and Scan Again.");
        deleteAuthFiles();
        sock.logout();
      } else if (reason === DisconnectReason.restartRequired) {
        console.log("Restart Required, Restarting...");
        connectToWhatsApp();
      } else if (reason === DisconnectReason.timedOut) {
        console.log("Connection TimedOut, Reconnecting...");
        connectToWhatsApp();
      } else {
        sock.end(`Unknown DisconnectReason: ${reason}|${connection}`);
      }
    } else if (connection === "open") {
      isConnected = true;
      updateQR("connected");
      console.log("WhatsApp Bot Connected Successfully!");
      console.log("WhatsApp Web.js Bot v2.0.0-Baileys started!");
      
      // Ambil semua grup dan update cache participant
      try {
        const groups = await sock.groupFetchAllParticipating();
        console.log(`Found ${Object.keys(groups).length} groups`);
        
        for (const groupId in groups) {
          const group = groups[groupId];
          console.log(`Group: ${group.subject} (${groupId})`);
          
          group.participants.forEach(participant => {
            const phoneNumber = participant.id.split('@')[0];
            const cacheKey = `${groupId}:${participant.id}`;
            participantCache.set(cacheKey, phoneNumber);
          });
        }
      } catch (error) {
        console.log("Error updating participant cache:", error.message);
      }
      
      return;
    }

    if (qr) {
      updateQR("qr");
    }
  });

    sock.ev.on("creds.update", saveCreds);
    
    sock.ev.on("messages.upsert", async ({ messages, type }) => {
      // console.log(messages);
      if (type === "notify") {
        const message = messages[0];
        const remoteJid = message.key.remoteJid;
        const isGroupMessage = /@g\.us$/.test(remoteJid);
        
        // Skip jika pesan dari diri sendiri
        if (message.key.fromMe) return;
        
        // Tandai pesan sebagai dibaca
        await sock.readMessages([message.key]);
        
        if (isGroupMessage) {
          await handleGroupMessage(message, sock);
        } else {
          await handlePrivateMessage(message, sock);
        }
      }
    });
    
  } catch (error) {
    console.error("Error connecting to WhatsApp:", error);
    setTimeout(() => connectToWhatsApp(), 5000);
  }
}

// Group message handler
async function handleGroupMessage(message, sock) {
    const remoteJid = message.key.remoteJid;
    // Hanya respons untuk grup tertentu
    const allowedGroups = [
      '120363183329351999@g.us', //WAbsensi UPTD PUSKESMAS MENSIKU
      '120363421159009634@g.us', //[NEW] Presensi - UPTD. PUSKESMAS MENSIKU
      '120363161304334042@g.us', //ProgSi - <ProgrammerSintang/> 
    ];
    if (!allowedGroups.includes(remoteJid)) {
      return;
    }
    const pesan = message.message?.conversation || message.message?.extendedTextMessage?.text;
    const pesanMasuk = typeof pesan === "string" ? pesan.toLowerCase() : "";
    const noWaParticipant = message.key.participant;
    const phoneNumber = noWaParticipant ? noWaParticipant.replace("@s.whatsapp.net", "") : "";
    const pushName = message.pushName || 'Unknown';
    
    console.log('=== GROUP MESSAGE ===');
    console.log(`Group ID: ${remoteJid}`);
    console.log(`Participant: ${noWaParticipant}`);
    console.log(`Push Name: ${pushName}`);
    console.log(`Message: ${pesanMasuk}`);
    console.log('====================');
    
    // // Simpan nama participant ke cache
    // if (noWaParticipant && pushName) {
    //     const participantNameKey = `${remoteJid}:${noWaParticipant}`;
    //     participantNamesCache.set(participantNameKey, pushName);
    //     console.log(`Menyimpan nama ke cache: ${noWaParticipant} -> ${pushName}`);
    // }
    
    // Command: ping
    if (pesanMasuk === 'ping') {
        const phoneInfo = phoneNumber ? `\nNomor HP: ${phoneNumber}` : '';
        const answer = `_Hai saya adalah B0t_ 🤖.\n_Senang bisa membantu_ v.2.0.0-Baileys - FROM panel 😊${phoneInfo}\nNama: ${pushName}`;
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // Command: absen
    if (pesanMasuk === 'absen') {
        const answer = `FROM GROUP -> Hai _*${pushName}*_ \nSilahkan kirim lokasi terkini untuk melakukan presensi kehadiran.`;
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // // Command: #infogrup
    // if (pesanMasuk === '#infogrup') {
    //     try {
    //         const groupMetadata = await sock.groupMetadata(remoteJid);
    //         const participants = groupMetadata.participants;
            
    //         let infoGrup = `*Info Grup: ${groupMetadata.subject}*\n\n`;
    //         infoGrup += `Jumlah Anggota: ${participants.length}\n\n`;
    //         infoGrup += `*Daftar Anggota:*\n`;
            
    //         for (let i = 0; i < participants.length; i++) {
    //             const participant = participants[i];
    //             const phoneNumber = participant.id.split('@')[0];
                
    //             // Cari nama dari cache berdasarkan participant ID
    //             const participantNameKey = `${remoteJid}:${participant.id}`;
    //             const cachedName = participantNamesCache.get(participantNameKey);
                
    //             const pushName = cachedName || 'Belum ada pesan';
    //             const adminStatus = participant.admin ? ' (admin)' : '';
    //             const deviceType = participant.id.includes(':0') ? ' 📞' : ' 📱';
                
    //             infoGrup += `${i + 1}. ${phoneNumber}${adminStatus}${deviceType}\n`;
    //             infoGrup += `    📝 *${pushName}*\n\n`;
    //         }
            
    //         infoGrup += `\n📱 = WhatsApp Web/Desktop\n📞 = WhatsApp Mobile\n\n`;
    //         infoGrup += `*Note:* Nama diambil dari pesan terakhir yang dikirim`;
            
    //         await sendReply(sock, remoteJid, infoGrup, message);
    //     } catch (error) {
    //         console.log("Error getting group info:", error.message);
    //         await sendReply(sock, remoteJid, "Gagal mendapatkan info grup.", message);
    //     }
    // }
    
    // Command: #cachenama
    // if (pesanMasuk === '#cachenama') {
    //     try {
    //         let cacheInfo = `*Cache Nama Participant*\n\n`;
    //         cacheInfo += `Total nama tersimpan: ${participantNamesCache.size}\n\n`;
            
    //         let count = 0;
    //         for (const [key, value] of participantNamesCache.entries()) {
    //             if (key.startsWith(remoteJid)) {
    //                 count++;
    //                 const participantId = key.replace(`${remoteJid}:`, '');
    //                 const shortId = participantId.split('@')[0];
    //                 cacheInfo += `${count}. ${shortId}\n    📝 *${value}*\n\n`;
    //             }
    //         }
            
    //         if (count === 0) {
    //             cacheInfo += `Belum ada nama yang tersimpan untuk grup ini.\nNama akan tersimpan ketika anggota mengirim pesan.`;
    //         }
            
    //         await sock.sendMessage(remoteJid, { text: cacheInfo });
    //     } catch (error) {
    //         console.log("Error getting cache info:", error.message);
    //         await sock.sendMessage(remoteJid, { text: "Gagal mendapatkan info cache." });
    //     }
    // }
    
    // // Command: #lokasilist
    // if (pesanMasuk === '#lokasilist') {
    //     try {
    //         let locationInfo = `*📍 Daftar Lokasi yang Diizinkan*\n\n`;
    //         locationInfo += `Radius yang diizinkan: 100 meter dari titik\n\n`;
    //         locationInfo += `*Koordinat Lokasi:*\n`;
            
    //         allowedLocations.forEach((location, index) => {
    //             locationInfo += `${index + 1}. ${location.name}\n`;
    //             locationInfo += `   📍 ${location.lat}, ${location.lon}\n\n`;
    //         });
            
    //         locationInfo += `*Cara menggunakan:*\n`;
    //         locationInfo += `1. Share live location dari WhatsApp\n`;
    //         locationInfo += `2. Bot akan mengecek apakah lokasi Anda dalam radius 100m dari salah satu titik di atas\n`;
    //         locationInfo += `3. Jika dalam radius, presensi akan diproses\n`;
    //         locationInfo += `4. Jika di luar radius, akan muncul pesan penolakan`;

    //         await sock.sendMessage(remoteJid, { text: locationInfo });
    //     } catch (error) {
    //         console.log("Error getting location info:", error.message);
    //         await sock.sendMessage(remoteJid, { text: "Gagal mendapatkan info lokasi." });
    //     }
    // }
    
    // Command untuk testing lokasi (admin only)
    // if (pesanMasuk.startsWith('#testlokasi') && allowedNumbers.includes(phoneNumber)) {
    //     const parts = pesanMasuk.split(" ");
    //     if (parts.length === 3) {
    //         const testLat = parseFloat(parts[1]);
    //         const testLon = parseFloat(parts[2]);
            
    //         if (!isNaN(testLat) && !isNaN(testLon)) {
    //             console.log(`Testing location: ${testLat}, ${testLon}`);
    //             const locationCheck = checkLocationInRadius(testLat, testLon, 100);
                
    //             let testResult = `*🧪 Test Lokasi*\n\n`;
    //             testResult += `📍 Koordinat Test: ${testLat}, ${testLon}\n\n`;
                
    //             if (locationCheck.isInRadius) {
    //                 testResult += `✅ *DALAM RADIUS*\n`;
    //                 testResult += `📍 Lokasi Terdekat: ${locationCheck.nearestLocation}\n`;
    //                 testResult += `📏 Jarak: ${locationCheck.distance} meter\n`;
    //             } else {
    //                 testResult += `❌ *LUAR RADIUS*\n`;
    //                 testResult += `📏 Semua lokasi di luar radius 100m\n\n`;
                    
    //                 // Tampilkan jarak ke semua lokasi
    //                 testResult += `*Jarak ke semua lokasi:*\n`;
    //                 allowedLocations.forEach((location, index) => {
    //                     const distance = calculateDistance(testLat, testLon, location.lat, location.lon);
    //                     testResult += `${index + 1}. ${location.name}: ${Math.round(distance)}m\n`;
    //                 });
    //             }
                
    //             testResult += `\n*Format:* #testlokasi [latitude] [longitude]`;
                
    //             await sendReply(sock, remoteJid, testResult, message);
    //         } else {
    //             await sendReply(sock, remoteJid, "Format salah. Gunakan: #testlokasi [latitude] [longitude]\nContoh: #testlokasi 0.19880519469882046 111.46750166076582", message);
    //         }
    //     } else {
    //         await sendReply(sock, remoteJid, "Format: #testlokasi [latitude] [longitude]\nContoh: #testlokasi 0.19880519469882046 111.46750166076582", message);
    //     }
    // }
    
    // Handle manual attendance command (format: #manual|tgl|jam), ambil latlang dari reply jika ada
    if (pesanMasuk.startsWith('#manual')) {
        console.log('_________________________ MANUAL PRESENSI ___________________________');
        console.log(`_________________________ NOMOR WA : ${phoneNumber} ___________________________`);

        try {
            // Only accept format: #manual|tgl|jam
            const parts = pesanMasuk.split("|");
            if (parts.length !== 3) {
                await sendReply(sock, remoteJid, "Format salah. Gunakan: #manual|tanggal|jamPresensi\nContoh: #manual|07/08/2025|07:52:00", message);
                return;
            }
            const tglAbsen = parts[1];
            const jamPresensi = parts[2];
            
            // Ambil latlang, caption, dan participantId dari quoted liveLocationMessage
            let latlang = "";
            let caption = "manual by admin";
            let participantId = null;
            
            const quotedMsg = message.message?.extendedTextMessage?.contextInfo?.quotedMessage;
            const quotedParticipant = message.message?.extendedTextMessage?.contextInfo?.participant;
            
            console.log("[DEBUG] quotedMessage:", JSON.stringify(quotedMsg));
            console.log("[DEBUG] quotedParticipant:", quotedParticipant);
            
            if (quotedMsg && quotedMsg.liveLocationMessage) {
                const lat = quotedMsg.liveLocationMessage.degreesLatitude;
                const lon = quotedMsg.liveLocationMessage.degreesLongitude;
                latlang = `${lat}, ${lon}`;
                caption = quotedMsg.liveLocationMessage.caption || caption;
                
                // Ambil participantId dari yang mengirim lokasi (quoted), bukan yang reply
                participantId = quotedParticipant || null;
                
                console.log(`[DEBUG] Data dari quoted message:`);
                console.log(`[DEBUG] - latlang: ${latlang}`);
                console.log(`[DEBUG] - caption: ${caption}`);
                console.log(`[DEBUG] - participantId: ${participantId}`);
            } else {
                latlang = "";
                console.log("[DEBUG] Tidak ada quoted liveLocationMessage, latlang kosong");
            }

            // Validasi: jika latlang kosong atau participantId tidak ada, beri warning ke user
            if (!latlang || !participantId) {
                let errorMsg = "Presensi manual harus reply ke pesan lokasi (live location) dari pegawai yang akan diabsen!";
                if (!latlang) errorMsg += "\n❌ Lokasi tidak ditemukan.";
                if (!participantId) errorMsg += "\n❌ Participant ID tidak ditemukan.";
                await sendReply(sock, remoteJid, errorMsg, message);
                return;
            }

            console.log(`[DEBUG] Final data untuk doPresenceToSheetManual:`);
            console.log(`[DEBUG] - tglAbsen: ${tglAbsen}`);
            console.log(`[DEBUG] - jamPresensi: ${jamPresensi}`);
            console.log(`[DEBUG] - participantId: ${participantId}`);
            console.log(`[DEBUG] - latlang: ${latlang}`);
            console.log(`[DEBUG] - caption: ${caption}`);
            
            const answer = await doPresenceToSheetManual(tglAbsen, jamPresensi, participantId, latlang, caption);
            await sendReply(sock, remoteJid, answer, message);
        } catch (error) {
            console.log("Error processing manual attendance:", error.message);
            await sendReply(sock, remoteJid, "Error memproses presensi manual. Pastikan format pesan benar.", message);
        }
    }
    
    // Handle location messages dengan geofencing
    if (message.message?.liveLocationMessage) {
        const caption = message.message?.liveLocationMessage?.caption || '';
        const userLat = message.message?.liveLocationMessage?.degreesLatitude;
        const userLon = message.message?.liveLocationMessage?.degreesLongitude;
        const latlang = `${userLat}, ${userLon}`;
        
        console.log("Live location processing...");
        console.log(`WA Number: ${phoneNumber}`);
        console.log(`User Location: ${latlang}`);
        console.log(`Push Name: ${pushName}`);
        console.log(`Group: ${remoteJid}`);
        
        // Cek apakah lokasi berada dalam radius yang diizinkan
        const locationCheck = checkLocationInRadius(userLat, userLon, 100);
        
        // if (!locationCheck.isInRadius) {
        //     // Lokasi tidak dalam radius - kirim pesan penolakan
        //     console.log("❌ Lokasi tidak dalam jangkauan area");
            
        //     const rejectMessage = `❌ Mohon Maaf anda tidak berada dalam jangkauan area.\n\n` +
        //                         `📝 *Detail Informasi:*\n` +
        //                         `ID Participant: ${noWaParticipant}\n` +
        //                         `Nama Participant: ${pushName}\n` +
        //                         `📍 Lokasi Anda: ${userLat}, ${userLon}\n` +
        //                         `💬 Caption: ${caption || 'Tidak ada caption'}\n\n` +
        //                         `ℹ️ Silakan pastikan Anda berada dalam radius 100m dari salah satu lokasi yang diizinkan. ` +
        //                         `Ketik #lokasilist untuk melihat daftar lokasi yang diizinkan.`;
            
        //     await sendReply(sock, remoteJid, rejectMessage, message);
            
        // } else {
            // Lokasi dalam radius - kirim konfirmasi dan proses presensi
            // console.log(`✅ Lokasi dalam radius di ${locationCheck.nearestLocation} (${locationCheck.distance}m)`);
            
            // const successMessage = `ID Participant: ${noWaParticipant}\n` +
            //                       `Nama Participant: ${pushName}\n` +
            //                       `💬 Caption: ${caption || 'Tidak ada caption'}\n` +
            //                       `Presensi berhasil, proses presensi dilakukan di background. Terimakasih\n` +
            //                       `📍 Lokasi: ${locationCheck.nearestLocation} (${locationCheck.distance}m dari titik)`;
            
            // await sendReply(sock, remoteJid, successMessage, message);
            
            // Background process - proses presensi ke Google Sheets
            try {
                console.log("🔄 Memproses presensi ke background...");
                const answer = await doPresenceToSheet("doPresence", phoneNumber, latlang, caption.toLowerCase());
                console.log("📋 Hasil presensi:", answer);
                
                // Reply hasil presensi dari Google Sheets
                if (answer) {
                    await sendReply(sock, remoteJid, answer, message);
                }
            } catch (error) {
                console.log("❌ Error processing presence:", error.message);
                // Reply error message
                await sendReply(sock, remoteJid, `❌ Error memproses presensi: ${error.message}`, message);
                
            }
        // }
    }
    
    // Hadis command
    const hadisPattern = /^#hadis\s(.+)$/;
    if (hadisPattern.test(pesanMasuk)) {
        let getOrigMsg = pesanMasuk.replace("#hadis ", "");
        let splitMsg = getOrigMsg.split(":");
        const answer = await getHadistApi(splitMsg[0], splitMsg[1]);
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // Quran command
    const quranPattern = /^#quran \d+:\d+$/;
    if (quranPattern.test(pesanMasuk)) {
        let getOrigMsg = pesanMasuk.replace("#quran ", "");
        let splitMsg = getOrigMsg.split(":");
        const answer = await getQuranApi(splitMsg[0], splitMsg[1]);
        await sendReply(sock, remoteJid, answer, message);
        
        // Send audio
        try {
            const audioUrl = await getQuranAudioApi(splitMsg[0], splitMsg[1]);
            if (audioUrl) {
                // Send audio as reply
                await sock.sendMessage(remoteJid, {
                    audio: { url: audioUrl },
                    mimetype: 'audio/mp4'
                }, { quoted: message });
            }
        } catch (error) {
            console.log('Error sending Quran audio:', error.message);
        }
    }
    
    // Sound command
    const soundPattern = /^#s\s(.+)$/;
    if (soundPattern.test(pesanMasuk)) {
        const qFromChat = pesanMasuk.replace("#s", "");
        let textsound = capital(qFromChat);
        let API_URL = `https://texttospeech.responsivevoice.org/v1/text:synthesize?text=${encodeURIComponent(textsound)}&lang=id&engine=g3&name=&pitch=0.5&rate=0.5&volume=1&key=kvfbSITh&gender=female`;
        
        try {
            await sock.sendMessage(remoteJid, {
                audio: { url: API_URL },
                mimetype: 'audio/mp4'
            }, { quoted: message });
        } catch (error) {
            console.log('Error generating sound:', error.message);
            await sendReply(sock, remoteJid, '❌ Gagal membuat audio. Silakan coba lagi.', message);
        }
    }
}

// Private message handler
async function handlePrivateMessage(message, sock) {
    const remoteJid = message.key.remoteJid;
    const pesan = message.message?.conversation || message.message?.extendedTextMessage?.text;
    const pesanMasuk = typeof pesan === "string" ? pesan.toLowerCase() : "";
    const phoneNumber = remoteJid.replace("@s.whatsapp.net", "");
    const pushName = message.pushName || 'Unknown';
    
    console.log('=== PRIVATE MESSAGE ===');
    console.log(`Phone Number: ${phoneNumber}`);
    console.log(`Push Name: ${pushName}`);
    console.log(`Message: ${pesanMasuk}`);
    console.log('=======================');
    
    // Command: ping
    if (pesanMasuk === 'ping') {
        await sendReply(sock, remoteJid, "_Hai saya adalah B0t_ 🤖.\n_Senang bisa membantu_ v2.0.0-Baileys - FROM panel 😊", message);
    }
    
    // Command: absen
    if (pesanMasuk === 'absen') {
        const answer = `Hai _*${pushName}*_ \nSilahkan kirim lokasi terkini untuk melakukan presensi kehadiran.`;
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // Command: #statpsb
    if (pesanMasuk === '#statpsb') {
        const answer = await getPSBStatistic();
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // Command: #hadislist
    if (pesanMasuk === '#hadislist') {
        const hadisLists = `\nFormat chatBot untuk hadis -> *HR. Abu Daud*\n#hadis abu-daud:1-4419\n\nFormat chatBot untuk hadis -> *HR. Ahmad*\n#hadis ahmad:1-4305\n\nFormat chatBot untuk hadis -> *HR. Bukhari*\n#hadis bukhari:1-6638\n\nFormat chatBot untuk hadis -> *HR. Darimi*\n#hadis darimi:1-2949\n\nFormat chatBot untuk hadis -> *HR. Ibnu Majah*\n#hadis ibnu-majah:1-4285\n\nFormat chatBot untuk hadis -> *HR. Malik*\n#hadis malik:1-1587\n\nFormat chatBot untuk hadis -> *HR. Muslim*\n#hadis muslim:1-4930\n\nFormat chatBot untuk hadis -> *HR. Nasai*\n#hadis nasai:1-5364\n\nFormat chatBot untuk hadis -> *HR. Tirmidzi*\n#hadis tirmidzi:1-3625`;
        await sendReply(sock, remoteJid, hadisLists, message);
    }
    
    // Command: #quranlist
    if (pesanMasuk === '#quranlist') {
        const answer = `*Qur'an List*\n\n1. Al-Fatihah, Pembukaan, 7 ayat سورة الفاتحة \n2. Al-Baqarah, Sapi Betina, 286 ayat سورة البقرة \n3. Ali 'Imran, Keluarga 'Imran, 200 ayat سورة آل عمران \n...dan seterusnya...`;
        await sendReply(sock, remoteJid, answer, message);
    }
    
    // // Command: #lokasilist
    // if (pesanMasuk === '#lokasilist') {
    //     try {
    //         let locationInfo = `*📍 Daftar Lokasi yang Diizinkan*\n\n`;
    //         locationInfo += `Radius yang diizinkan: 100 meter dari titik\n\n`;
    //         locationInfo += `*Koordinat Lokasi:*\n`;
            
    //         allowedLocations.forEach((location, index) => {
    //             locationInfo += `${index + 1}. ${location.name}\n`;
    //             locationInfo += `   📍 ${location.lat}, ${location.lon}\n\n`;
    //         });
            
    //         locationInfo += `*Cara menggunakan:*\n`;
    //         locationInfo += `1. Share live location dari WhatsApp\n`;
    //         locationInfo += `2. Bot akan mengecek apakah lokasi Anda dalam radius 100m dari salah satu titik di atas\n`;
    //         locationInfo += `3. Jika dalam radius, presensi akan diproses\n`;
    //         locationInfo += `4. Jika di luar radius, akan muncul pesan penolakan`;

    //         await sendReply(sock, remoteJid, locationInfo, message);
    //     } catch (error) {
    //         console.log("Error getting location info:", error.message);
    //         await sendReply(sock, remoteJid, "Gagal mendapatkan info lokasi.", message);
    //     }
    // }
    
    // Handle location for private chat dengan geofencing
    if (message.message?.liveLocationMessage) {
        const caption = message.message?.liveLocationMessage?.caption || '';
        const userLat = message.message?.liveLocationMessage?.degreesLatitude;
        const userLon = message.message?.liveLocationMessage?.degreesLongitude;
        const latlang = `${userLat}, ${userLon}`;
        
        // console.log("Live location processing (private message)...");
        // console.log(`WA Number: ${phoneNumber}`);
        // console.log(`User Location: ${latlang}`);
        // console.log(`Push Name: ${pushName}`);
        
        // // Cek apakah lokasi berada dalam radius yang diizinkan
        // const locationCheck = checkLocationInRadius(userLat, userLon, 100);
        
        // if (!locationCheck.isInRadius) {
            // Lokasi tidak dalam radius - kirim pesan penolakan
            // console.log("❌ Lokasi tidak dalam jangkauan area (private message)");
            
            // const rejectMessage = `❌ Mohon Maaf anda tidak berada dalam jangkauan area.\n\n` +
            //                     `📝 *Detail Informasi:*\n` +
            //                     `ID Participant: ${remoteJid}\n` +
            //                     `Nama Participant: ${pushName}\n` +
            //                     `📍 Lokasi Anda: ${userLat}, ${userLon}\n` +
            //                     `💬 Caption: ${caption || 'Tidak ada caption'}\n\n` +
            //                     `ℹ️ Silakan pastikan Anda berada dalam radius 100m dari salah satu lokasi yang diizinkan. ` +
            //                     `Ketik #lokasilist untuk melihat daftar lokasi yang diizinkan.`;
            
            // await sendReply(sock, remoteJid, rejectMessage, message);
            
        // } else {
            // // Lokasi dalam radius - kirim konfirmasi dan proses presensi
            // console.log(`✅ Lokasi dalam radius di ${locationCheck.nearestLocation} (${locationCheck.distance}m) - private message`);
            
            // const successMessage = `ID Participant: ${remoteJid}\n` +
            //                       `Nama Participant: ${pushName}\n` +
            //                       `💬 Caption: ${caption || 'Tidak ada caption'}\n` +
            //                       `Presensi berhasil, proses presensi dilakukan di background. Terimakasih\n` +
            //                       `📍 Lokasi: ${locationCheck.nearestLocation} (${locationCheck.distance}m dari titik)`;
            
            // await sendReply(sock, remoteJid, successMessage, message);
            
            // Background process - proses presensi ke Google Sheets
            try {                
                console.log("🔄 Memproses presensi ke background (private message)...");
                const answer = await doPresenceToSheet("doPresence", phoneNumber, latlang, caption.toLowerCase());
                console.log("📋 Hasil presensi:", answer);
                
                // Reply hasil presensi dari Google Sheets
                if (answer) {
                    await sendReply(sock, remoteJid, answer, message);
                }
            } catch (error) {
                console.log("❌ Error processing presence:", error.message);
                // Reply error message
                await sendReply(sock, remoteJid, `❌ Error memproses presensi: ${error.message}`, message);
            }
        // }
    }
}

// API Routes
app.get("/check-wa-number", async (req, res) => {
  const getIdKey = req.query.idkey;
  const waNumber = req.query.wanumber;
  
  if (getIdKey == CONFIG.API_KEY) {
    try {
      const numberWA = "62" + waNumber.substring(1) + "@s.whatsapp.net";
      if (isConnected) {
        const exists = await sock.onWhatsApp(numberWA);
        console.log([exists]);
        if (exists?.jid || (exists && exists[0]?.jid)) {
          res.status(200).json({
            status: true,
            response: `Nomor ${waNumber} valid.`,
          });
        } else {
          res.status(500).json({
            status: false,
            response: `Nomor ${waNumber} tidak terdaftar.`,
          });
        }
      } else {
        res.status(500).json({
          status: false,
          response: `WhatsApp belum terhubung.`,
        });
      }
    } catch (err) {
      console.log(err);
      res.status(500).send(err);
    }
  } else {
    res.status(500).json({
      status: false,
      response: `Upppss...`,
    });
  }
});

app.get("/pm", async (req, res) => {
  const getIdKey = req.query.idkey;
  const waNumber = req.query.wanumber;
  const message = req.query.message;
  
  if (getIdKey == CONFIG.API_KEY) {
    try {
      const numberWA = "62" + waNumber.substring(1) + "@s.whatsapp.net";
      if (isConnected) {
        const exists = await sock.onWhatsApp(numberWA);
        console.log(exists);
        if (exists?.jid || (exists && exists[0]?.jid)) {
          sock.sendMessage(exists.jid || exists[0].jid, { text: message })
            .then((result) => {
              res.status(200).json({
                status: true,
                response: result,
              });
            })
            .catch((err) => {
              res.status(500).json({
                status: false,
                response: err,
              });
            });
        } else {
          res.status(500).json({
            status: false,
            response: `Nomor ${waNumber} tidak terdaftar.`,
          });
        }
      } else {
        res.status(500).json({
          status: false,
          response: `WhatsApp belum terhubung.`,
        });
      }
    } catch (err) {
      console.log(err);
      res.status(500).send(err);
    }
  } else {
    res.status(500).json({
      status: false,
      response: `Upppss... tidak valid!`,
    });
  }
});

app.get("/check-status", async (req, res) => {
    res.json({
        connected: isConnected,
        timestamp: new Date().toISOString()
    });
});

app.post("/send-message", async (req, res) => {
  const pesankirim = req.body.message;
  const number = req.body.number;
  const fileDikirim = req.files;
  
  try {
    if (!req.files) {
      if (!number) {
        res.status(500).json({
          status: false,
          response: "Nomor WA belum tidak disertakan!",
        });
      } else {
        const numberWA = "62" + number.substring(1) + "@s.whatsapp.net";
        
        if (isConnected) {
          const exists = await sock.onWhatsApp(numberWA);
          if (exists?.jid || (exists && exists[0]?.jid)) {
            await sock.sendMessage(exists.jid || exists[0].jid, { text: pesankirim });
            res.status(200).json({
              status: true,
              response: "Message sent successfully",
            });
          } else {
            res.status(500).json({
              status: false,
              response: `Nomor ${number} tidak terdaftar.`,
            });
          }
        } else {
          res.status(500).json({
            status: false,
            response: `WhatsApp belum terhubung.`,
          });
        }
      }
    } else {
      // Handle file upload
      if (!number) {
        res.status(500).json({
          status: false,
          response: "Nomor WA belum tidak disertakan!",
        });
      } else {
        const numberWA = "62" + number.substring(1) + "@s.whatsapp.net";
        let filesimpan = req.files.file_dikirim;
        var file_ubah_nama = new Date().getTime() + "_" + filesimpan.name;
        filesimpan.mv("./uploads/" + file_ubah_nama);
        
        if (isConnected) {
          const exists = await sock.onWhatsApp(numberWA);
          if (exists?.jid || (exists && exists[0]?.jid)) {
            let namafiledikirim = "./uploads/" + file_ubah_nama;
            let extensionName = path.extname(namafiledikirim);
            
            const fileBuffer = fs.readFileSync(namafiledikirim);
            
            if (extensionName === '.jpg' || extensionName === '.jpeg' || extensionName === '.png') {
              await sock.sendMessage(exists.jid || exists[0].jid, {
                image: fileBuffer,
                caption: pesankirim
              });
            } else if (extensionName === '.mp4' || extensionName === '.avi' || extensionName === '.mov') {
              await sock.sendMessage(exists.jid || exists[0].jid, {
                video: fileBuffer,
                caption: pesankirim
              });
            } else if (extensionName === '.mp3' || extensionName === '.wav' || extensionName === '.ogg') {
              await sock.sendMessage(exists.jid || exists[0].jid, {
                audio: fileBuffer,
                mimetype: 'audio/mp4'
              });
            } else {
              await sock.sendMessage(exists.jid || exists[0].jid, {
                document: fileBuffer,
                fileName: filesimpan.name,
                mimetype: filesimpan.mimetype
              });
            }
            
            // Clean up
            if (fs.existsSync(namafiledikirim)) {
              fs.unlink(namafiledikirim, (err) => {
                if (err) console.log(err);
              });
            }
            
            res.send({
              status: true,
              message: "Success",
              data: {
                name: filesimpan.name,
                mimetype: filesimpan.mimetype,
                size: filesimpan.size,
              },
            });
          } else {
            res.status(500).json({
              status: false,
              response: `Nomor ${number} tidak terdaftar.`,
            });
          }
        } else {
          res.status(500).json({
            status: false,
            response: `WhatsApp belum terhubung.`,
          });
        }
      }
    }
  } catch (err) {
    res.status(500).send(err);
  }
});

// Socket.IO for web interface
io.on("connection", async (socket) => {
    soket = socket;
    if (isConnected) {
        updateQR("connected");
    } else if (qrDinamic) {
        updateQR("qr");
    }
});

const updateQR = async (data) => {
    switch (data) {
        case "qr":
            let rawData = await qrcode.toDataURL(qrDinamic, { scale: 8 });
            setTimeout(() => {
                soket?.emit("qr", rawData);
                soket?.emit("log", "QR Code received, please scan!");
            }, 2000);
            break;
        case "connected":
            soket?.emit("log", "WhatsApp terhubung!");
            break;
        case "qrscanned":
            soket?.emit("qrstatus", "./assets/check.svg");
            soket?.emit("log", "QR Code Telah discan!");
            break;
        case "loading":
            if (typeof qrDinamic === "string") {
                qrcode.toDataURL(qrDinamic, (err, url) => {
                    if (err) {
                        console.error("Error generating QR code:", err);
                        return;
                    }
                    soket?.emit("qr", url);
                    soket?.emit("log", "QR Code received, please scan!");
                });
            } else {
                console.error("QR data is not a string:", qrDinamic);
            }
            break;
        default:
            break;
    }
};

// Utility functions
function capital(textSound) {
    const arr = textSound.split(" ");
    for (var i = 0; i < arr.length; i++) {
        arr[i] = arr[i].charAt(0).toUpperCase() + arr[i].slice(1);
    }
    const str = arr.join(" ");
    return str;
}

// Helper function untuk reply message
async function sendReply(sock, remoteJid, replyMessage, originalMessage) {
    try {
        await sock.sendMessage(remoteJid, {
            text: replyMessage
        }, {
            quoted: originalMessage
        });
    } catch (error) {
        console.log("Error sending reply:", error.message);
        // Fallback: send normal message if reply fails
        await sock.sendMessage(remoteJid, { text: replyMessage });
    }
}

// Fungsi untuk menghitung jarak antara dua titik koordinat menggunakan Haversine formula
function calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371000; // Radius bumi dalam meter
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLon = (lon2 - lon1) * Math.PI / 180;
    const a = 
        Math.sin(dLat/2) * Math.sin(dLat/2) +
        Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * 
        Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    const distance = R * c; // Jarak dalam meter
    return distance;
}

// Fungsi untuk mengecek apakah lokasi berada dalam radius yang diizinkan
function checkLocationInRadius(userLat, userLon, radiusInMeters = 100) {
    for (let location of allowedLocations) {
        const distance = calculateDistance(userLat, userLon, location.lat, location.lon);
        if (distance <= radiusInMeters) {
            return {
                isInRadius: true,
                nearestLocation: location.name,
                distance: Math.round(distance)
            };
        }
    }
    return {
        isInRadius: false,
        nearestLocation: null,
        distance: null
    };
}

async function doPresenceToSheet(presenceType, waNumber, latlang, caption) {
    try {
        const deployId = CONFIG.GOOGLE_SHEETS_DEPLOY_ID;
        const { data } = await axios.get(
            `https://script.google.com/macros/s/${deployId}/exec?action=${presenceType}&waNumber=${waNumber}&latlang=${latlang}&caption=${encodeURIComponent(caption)}&platform=fromWhatsapp`,
            {
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                },
                timeout: 30000
            }
        );

        console.log(">>>__ JAWABAN :" + JSON.stringify(data));

        if (data.success == true) {
            var answer = `✅ Presensi berhasil\n----------------------------------------------------------------\n👨‍💼 _Nama Pegawai_: *${data.presence.name}*\n📆 _Tanggal_: *${data.presence.date}*\n⏱️ _Jam Presensi_: *${data.presence.jam_datang}*\n📝 _Keterangan_: *${data.presence.caption}*\n----------------------------------------------------------------\n\n*🏥 ${data.presence.brand}*`;
        } else {
            var answer = `❌ ${data.message}\n---------------------------------------------------------------\n\n*🏥 ${data.brand}*`;
        }
        return answer;
    } catch (error) {
        if (axios.isAxiosError(error)) {
            console.log("PRESENSI ERROR : error message: ", error.message);
            return error.message;
        } else {
            console.log("unexpected error: ", error);
            return "PRESENSI ERROR : An unexpected error occurred";
        }
    }
}

async function doPresenceToSheetManual(tglAbsen, jamPresensi, participantId, latlang, caption) {
    try {
        const deployId = CONFIG.GOOGLE_SHEETS_DEPLOY_ID;
        
        // Buat URL dengan participantId (tidak ada duplikasi)
        const url = `https://script.google.com/macros/s/${deployId}/exec?participantId=${encodeURIComponent(participantId)}&tanggal=${tglAbsen}&jamPresensi=${jamPresensi}&koordinat=${encodeURIComponent(latlang)}&keterangan=${encodeURIComponent(caption)}&metodeabsen=manual`;
        
        console.log("[DEBUG] Final URL:", url);
        
        const { data } = await axios.get(url, {
            headers: {
                "Content-Type": "application/json",
                Accept: "application/json",
            },
            timeout: 30000
        });

        console.log(">>>__ JAWABAN :" + JSON.stringify(data));

        if (data.status == "success") {
            var answer = `✅ Presensi berhasil\n\n👨‍💼 _Nama Pegawai_: *${data.data.namaPegawai}*\n📆 _Tanggal_: *${data.data.tanggal}*\n⏱️ _Jam Presensi_: *${data.data.jamPresensi}*\n📝 _Keterangan_: *${data.data.keterangan}*\n\n*${data.message}*`;
        } else {
            var answer = `❌ ${data.message}`;
        }
        return answer;
    } catch (error) {
        if (axios.isAxiosError(error)) {
            console.log("PRESENSI ERROR : error message: ", error.message);
            return error.message;
        } else {
            console.log("unexpected error: ", error);
            return "PRESENSI ERROR : An unexpected error occurred";
        }
    }
}

async function getHadistApi(book, number) {
    try {
        const { data } = await axios.get(
            `https://api.hadith.gading.dev/books/${book}/${number}`,
            {
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                },
            }
        );

        console.log(">>>__ JAWABAN :" + JSON.stringify(data.data));
        const hadistText = `${data.data.name}: ${data.data.contents.number} \n\n ${data.data.contents.arab}\n\n ${data.data.contents.id}`;
        return hadistText;
    } catch (error) {
        if (axios.isAxiosError(error)) {
            console.log("HADIST API ERROR : error message: ", error.message);
            return error.message;
        } else {
            console.log("unexpected error: ", error);
            return "HADIST API ERROR : An unexpected error occurred";
        }
    }
}

async function getQuranApi(surah, ayah) {
    try {
        const { data } = await axios.get(
            `https://api.quran.gading.dev/surah/${surah}/${ayah}`,
            {
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                },
            }
        );

        const quranText = `QS. ${surah}:${ayah}\n*${data.data.surah.name.transliteration.id} ayat ${ayah}*\n${data.data.surah.name.long}\n\n${data.data.text.arab} (${ayah})\n${data.data.translation.id} (${ayah})`;
        console.log(quranText);
        return quranText;
    } catch (error) {
        console.log("unexpected error: ", error);
        return "Error getting Quran verse";
    }
}

async function getQuranAudioApi(surah, ayah) {
    try {
        const { data } = await axios.get(
            `https://api.quran.gading.dev/surah/${surah}/${ayah}`,
            {
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                },
            }
        );
        return data.data.audio.primary;
    } catch (error) {
        console.log("Error getting Quran audio:", error);
        return null;
    }
}

async function getPSBStatistic() {
    try {
        // Implement your PSB statistic logic here
        return "PSB Statistics not implemented yet";
    } catch (error) {
        console.log("Error getting PSB statistics:", error);
        return "Error getting PSB statistics";
    }
}

// Error handling untuk production
process.on('unhandledRejection', (reason, promise) => {
    console.error('🚨 Unhandled Rejection at:', promise, 'reason:', reason);
    // Log but don't crash in production
});

process.on('uncaughtException', (error) => {
    console.error('🚨 Uncaught Exception:', error);
    if (CONFIG.NODE_ENV !== 'production') {
        process.exit(1);
    }
});

// Graceful shutdown
process.on('SIGINT', () => {
    console.log('🛑 Shutting down gracefully...');
    if (sock) {
        sock.end();
    }
    server.close(() => {
        console.log('✅ Server closed');
        process.exit(0);
    });
});

// Memory monitoring for cPanel
if (CONFIG.CPANEL_MODE) {
    setInterval(() => {
        const memUsage = process.memoryUsage();
        const memUsageMB = Math.round(memUsage.heapUsed / 1024 / 1024);
        console.log(`💾 Memory usage: ${memUsageMB}MB`);
    }, 300000); // Every 5 minutes
}

// Initialize WhatsApp client dan start server
async function startBot() {
    try {
        console.log(`🚀 Starting ${CONFIG.BOT_NAME} ${CONFIG.BOT_VERSION}`);
        console.log(`🌍 Environment: ${CONFIG.NODE_ENV}`);
        console.log(`🏗️  cPanel Mode: ${CONFIG.CPANEL_MODE ? 'Enabled' : 'Disabled'}`);
        
        await connectToWhatsApp();
        
        server.listen(port, () => {
            console.log(`✅ Server running on port: ${port}`);
            console.log(`🤖 ${CONFIG.BOT_NAME} ${CONFIG.BOT_VERSION} (cPanel Compatible) started!`);
            console.log(`🌐 Web interface: http://localhost:${port}`);
            console.log(`📱 QR Scanner: http://localhost:${port}/scan`);
        });
        
    } catch (error) {
        console.error('❌ Failed to start bot:', error);
        setTimeout(startBot, 5000);
    }
}

startBot();
